// This source code is subject to the terms of the Mozilla Public License 2.0 at https://mozilla.org/MPL/2.0/
// © Barnabash

// Indicators: Parabolic SAR, DMI, EMA, Schaff Trend Cycle

// Buy Rules:
// 1. Parabolic SAR Indicates UpTrend.
// 2. ADX (Red Line) above the preferred value.
// 3. The +DI (Blue Line) crosses above the -DI (Orange Line).
// 4. The signal strengthens if ADX rises.
// 5. Bars above EMA.
// 6. Schaff Trend Cycle Rising up Indicating an UpTrend.

// Sell Rules: 
// 1. Parabolic SAR Indicates DownTrend.
// 2. ADX (Red Line) under the preferred value.
// 3. The +DI (Blue Line) crosses below the -DI (Orange Line).
// 4. The signal strengthens if ADX falls.
// 5. Bars below EMA.
// 6. Schaff Trend Cycle Falls down Indicating a DownTrend.

//@version = 5
import TradingView/ta/5

strategy(title = 'BTC Strategy',
     shorttitle = 'BBTC',
     overlay=false,
     pyramiding = 1,
     default_qty_type = strategy.percent_of_equity,
     default_qty_value = 100,
     initial_capital = 500,
     currency = currency.USD)

// User Inputs
var g_parabolic     = 'Parabolic Stop and Reverse (SAR)'
i_sarstart          = input.float(title = 'Parabolic SAR Start Value', defval = 0.02, step = 0.01, tooltip = 'Set your preferred SAR Start value.', group = g_parabolic)
i_sarincrement      = input.float(title = 'Parabolic SAR Increment Value', defval = 0.02, step = 0.01, tooltip = 'Set your preferred SAR Increment value.', group = g_parabolic)
i_sarmaximum        = input.float(title = 'Parabolic SAR Maximum value', defval = 0.2, step = 0.01, tooltip = 'Set your preferred SAR Maximum value.', group = g_parabolic)

var g_dmi           = 'Directional Moving Index (DMI)'
i_dmilenght         = input.int(title = 'DMI Lenght', defval = 20, tooltip = 'Set your preferred DMI Lenght.', minval = 1, group = g_dmi)
i_adxsmoothing      = input.int(title = 'ADX Smoothing', defval = 30, tooltip = 'Set your preferred ADX Smoothing value.', minval = 1, group = g_dmi)

var g_ema           = 'Exponential Moving Average (EMA)'
i_emalenght         = input.int(title = 'EMA Lenght', defval = 14, tooltip = 'Set your Preferred EMA Lenght.', minval = 1, group = g_ema)

var g_schaff        = 'Schaff Trend Cycle'
fastLength          = input.int(title="MACD Fast Length", defval=23, tooltip = 'Set your preferred Fast Lenght.', group = g_schaff)
slowLength          = input.int(title="MACD Slow Length", defval=50, tooltip = 'Set your preferred Slow Lenght.', group = g_schaff)
cycleLength         = input.int(title="Cycle Length", defval=10, tooltip = 'Set your preferred Cycle Lenght.', group = g_schaff)
d1Length            = input.int(title="1st %D Length", defval=3, tooltip = 'Set your preferred 1st %D Lenght.', group = g_schaff)
d2Length            = input.int(title="2nd %D Length", defval=3, tooltip = 'Set your preferred 2nd %D Lenght.', group = g_schaff)
upper               = input.int(title="Upper Band", defval=75, tooltip = 'Set your preferred Upper Band value.', group = g_schaff)
lower               = input.int(title="Lower Band", defval=25, tooltip = 'Set your preferred Lowe Band value.', group = g_schaff)

// Get Parabolic SAR Values
SAR = ta.sar(i_sarstart, i_sarincrement, i_sarmaximum)

// Get DMI Values
[diplus, diminus, adx] = ta.dmi(i_dmilenght, i_adxsmoothing)

// Get EMA Values
EMA = ta.ema(close, i_emalenght)

// Get Schaff Trend Cycle Values 
macd = ta.ema(close, fastLength) - ta.ema(close, slowLength)
k = nz(fixnan(ta.stoch(macd, macd, macd, cycleLength)))
d = ta.ema(k, d1Length)
kd = nz(fixnan(ta.stoch(d, d, d, cycleLength)))
stc = ta.ema(kd, d2Length)

// Get Buy Signal
BuySignal = barstate.isconfirmed and SAR < close and adx > 25 and ta.crossover(diplus, diminus) and close > EMA and ta.crossover(stc, lower)

// Get Sell Signal
SellSignal = barstate.isconfirmed and SAR > close and adx < 25 and ta.crossunder(diplus, diminus) and close > EMA and ta.crossunder(stc, upper)

// Strategy Enters and Closes
if BuySignal
     strategy.entry(id = 'Buy', direction = strategy.long, alert_message = 'Buy Order Triggered.')
     strategy.close(id = 'Buy', when = SellSignal, alert_message = 'Buy Order Close Triggered.')

// Coba Metrics Table
import EliCobra/CobraMetrics/4 as cobra

i_disp_ind = input.string ("None" , title = "Display Curve" , tooltip = "Choose which data you would like to display", options=["Strategy", "Equity", "Open Profit", "Gross Profit", "Net Profit", "None"], group = "🐍 𝓒𝓸𝓫𝓻𝓪 𝓜𝓮𝓽𝓻𝓲𝓬𝓼 🐍")
i_pos_table = input.string("Middle Left", "Table Position", options = ["Top Left", "Middle Left", "Bottom Left", "Top Right", "Middle Right", "Bottom Right", "Top Center", "Bottom Center"], group = "🐍 𝓒𝓸𝓫𝓻𝓪 𝓜𝓮𝓽𝓻𝓲𝓬𝓼 🐍")
i_type_table = input.string("None", "Table Type", options = ["Full", "Simple", "None"], group = "🐍 𝓒𝓸𝓫𝓻𝓪 𝓜𝓮𝓽𝓻𝓲𝓬𝓼 🐍")

plot(cobra.curve(i_disp_ind))
cobra.cobraTable(i_type_table, i_pos_table)